
  /*
   *  Object %name    : main.c%
   *  State           :  %state%
   *  Creation date   :  Wed Nov 17 17:39:24 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief This file includes the RSA parsing functions  
   *
   *  \version parser.c#1:csrc:1
   *  \author mottig
   *  \remarks Copyright (C) 2004 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/

#include "DX_VOS_BaseTypes.h"
#include "extapp_err.h"
#include "sep_utils.h"
#include "CRYS_RSA_Types.h"
#include "crysatptest_msg_defs.h"
#include "CRYS_SEP_ATP_Defines.h"
#include "DX_VOS_Mem.h"
#include "CRYS_SEP_ATP_RSA_Acceptance.h"



/************************ Defines ****************************/
/************************ Enums ******************************/
/************************ Typedefs ***************************/
/************************ Global Data ************************/
extern DxUint8_t   FailureFunctionNameOut[TST_FUNC_STRING_LEN];
/**************** Private function prototypes ****************/

/*****************************************************************************
* Function Name:                                                           
*  DX_SEP_PARSER_RSA_PRIME_EncDecTest

*  
* Inputs:
*  msg - message with all requested parameters.
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
* This function does the parsing for the external application that executes
* RSA PRIME encrypt/decrypt test.
* It calls the SEP function with parameters parsed from input message
* Algorithm: 
* 1. extract parameters from the message 
* 2. call QA SEP function
* 3. end SEP message and return
******************************************************************************/                                

DxUint32_t DX_SEP_PARSER_RSA_PRIME_EncDecTest(DxUint32_t* msg)
{

	/********* Variables Definitions ************/
	DxUint32_t					SEP_Error = DX_SUCCESS;
	/* input data size in bytes: */
	DxUint16_t					SEP_DataInSize = 0; 
	/* input data size word aligned */
	DxUint32_t					SEP_MaxDataInSize = RSA_BufferMaxLength; 
	/* message length in words */
	DxUint32_t					SEP_MsgLengthInWords;

	DxUint8_t	                *SEP_PubExponent_ptr;
	DxUint16_t					 SEP_PubExponentSize = 0;
	DxUint16_t					 SEP_MaxPubExponentSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_PrivExponent_ptr;
	DxUint16_t					 SEP_PrivExponentSize;
	DxUint16_t					 SEP_MaxPrivExponentSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_Modulus_ptr;
	
	/* modulus size */
	DxUint16_t					 SEP_KeySize;
	DxUint16_t					 SEP_MaxKeySize = RSA_BufferMaxLength; 
	DxUint8_t					 SEP_CRT_Mode;
	DxUint8_t					*SEP_P_ptr; 
	DxUint16_t					 SEP_PSize;
	DxUint16_t					 SEP_MaxPSize = RSA_BufferMaxLength; //CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8; 
	DxUint8_t					*SEP_Q_ptr;
	DxUint16_t					 SEP_QSize;
	DxUint16_t					 SEP_MaxQSize = RSA_BufferMaxLength;//CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8; 
	DxUint8_t					*SEP_dP_ptr;
	DxUint16_t					 SEP_dPSize;
	DxUint16_t					 SEP_MaxdPSize = RSA_BufferMaxLength;//CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8; 
	DxUint8_t					*SEP_dQ_ptr;
	DxUint16_t					 SEP_dQSize;
	DxUint16_t					 SEP_MaxdQSize = RSA_BufferMaxLength;//CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8; 
	DxUint8_t					*SEP_qInv_ptr;
	DxUint16_t					 SEP_qInvSize;
	DxUint16_t					 SEP_MaxqInvSize = RSA_BufferMaxLength;//CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8; 
	DxUint8_t					*SEP_Data_ptr;
	DxUint8_t					*SEP_ExpectedData_ptr;
	DxUint16_t					 SEP_MaxExpectedDataSize = RSA_BufferMaxLength;//CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8; 
	
	/* 1. extract parameters from the message */
	/******************************************/
	SEP_DataInSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_DATASIZE_IN_OFFSET];

	SEP_PubExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET];
	SEP_PrivExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET];
	/* modulus size */
	SEP_KeySize     = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_KEYSIZE_OFFSET];
	SEP_CRT_Mode    = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_CRTMODE_OFFSET];
	SEP_PSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PSIZE_OFFSET];
	SEP_QSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_QSIZE_OFFSET];
	SEP_dPSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dPSIZE_OFFSET];
	SEP_dQSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dQSIZE_OFFSET];
	SEP_qInvSize    = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_qInvSIZE_OFFSET];

	/* message length in words = length of the not buffer parameters */
	/* + all buffers parameters maximal lengthes                     */
	SEP_MsgLengthInWords = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS + 
		SEP_MaxDataInSize/4 + SEP_MaxPubExponentSize/4 + SEP_MaxPrivExponentSize/4 +
		SEP_MaxKeySize/4 + SEP_MaxPSize/4 + SEP_MaxQSize/4 + SEP_MaxdPSize/4 + 
		SEP_MaxdQSize/4 + SEP_MaxqInvSize/4 + SEP_MaxExpectedDataSize/4;

	SEP_PubExponent_ptr  = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PubExponent_OFFSET];

	SEP_PrivExponent_ptr = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PrivExponent_OFFSET];
	SEP_Modulus_ptr		 = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Modulus_OFFSET];
	SEP_P_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_P_OFFSET];
	SEP_Q_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Q_OFFSET];
	SEP_dP_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dP_OFFSET];
	SEP_dQ_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dQ_OFFSET];
	SEP_qInv_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_qInv_OFFSET];
	SEP_Data_ptr           = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_DataIn_OFFSET];
	SEP_ExpectedData_ptr   = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_DataExpected_OFFSET];


	/* 2. call QA SEP function */
	/***************************/
	SEP_Error = SEPQA_RSA_PRIME_EncDec (SEP_PubExponent_ptr,
										SEP_PubExponentSize,
										SEP_PrivExponent_ptr,
										SEP_PrivExponentSize,
										SEP_Modulus_ptr,
										SEP_KeySize,
										SEP_CRT_Mode,
										SEP_P_ptr, 
										SEP_PSize,
										SEP_Q_ptr,
										SEP_QSize,
										SEP_dP_ptr, 
										SEP_dPSize,
										SEP_dQ_ptr,
										SEP_dQSize,
										SEP_qInv_ptr,
										SEP_qInvSize,
										SEP_Data_ptr,
										SEP_DataInSize,
										SEP_ExpectedData_ptr,
										FailureFunctionNameOut);

	if(SEP_Error != CRYS_OK)
	{
		/* output parameters are placed after msg RC */
		DX_VOS_FastMemCpy((DxUint8_t  *)&msg[DX_SEP_HOST_SEP_PROTOCOL_MSG_RC_OFFSET + 1], (DxUint8_t  *)&FailureFunctionNameOut[0], TST_FUNC_STRING_LEN);
		SEP_MsgLengthInWords = TST_FUNC_STRING_LEN/sizeof(DxUint32_t);
	}

	/* 3. end SEP message and return */
	/**********************/
	return DX_SEP_EndMessage(msg, SEP_Error, SEP_MsgLengthInWords*sizeof(DxUint32_t));
}


/*****************************************************************************
* Function Name:                                                           
*  DX_SEP_PARSER_RSA_PRIME_SignVerifyTest

*  
* Inputs:
*  msg - message with all requested parameters.
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
* This function does the parsing for the external application that executes
* RSA PRIME encrypt/decrypt test.
* It calls the SEP function with parameters parsed from input message
* Algorithm: 
* 1. extract parameters from the message 
* 2. call QA SEP function
* 3. end SEP message and return
******************************************************************************/                                

DxUint32_t DX_SEP_PARSER_RSA_PRIME_SignVerifyTest(DxUint32_t* msg)
{

	/********* Variables Definitions ************/
	DxUint32_t					SEP_Error = DX_SUCCESS;
	/* input data size in bytes: */
	DxUint16_t					SEP_DataInSize = 0; 
	/* input data size word aligned */
	DxUint32_t					SEP_MaxDataInSize = RSA_BufferMaxLength; 
	/* message length in words */
	DxUint32_t					SEP_MsgLengthInWords;

	DxUint8_t	                *SEP_PubExponent_ptr;
	DxUint16_t					 SEP_PubExponentSize = 0;
	DxUint16_t					 SEP_MaxPubExponentSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_PrivExponent_ptr;
	DxUint16_t					 SEP_PrivExponentSize;
	DxUint16_t					 SEP_MaxPrivExponentSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_Modulus_ptr;

	/* modulus size */
	DxUint16_t					 SEP_KeySize;
	DxUint16_t					 SEP_MaxKeySize = RSA_BufferMaxLength; 
	DxUint8_t					 SEP_CRT_Mode;
	DxUint8_t					*SEP_P_ptr; 
	DxUint16_t					 SEP_PSize;
	DxUint16_t					 SEP_MaxPSize = RSA_BufferMaxLength; //CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8; 
	DxUint8_t					*SEP_Q_ptr;
	DxUint16_t					 SEP_QSize;
	DxUint16_t					 SEP_MaxQSize = RSA_BufferMaxLength;//CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8; 
	DxUint8_t					*SEP_dP_ptr;
	DxUint16_t					 SEP_dPSize;
	DxUint16_t					 SEP_MaxdPSize = RSA_BufferMaxLength;//CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8; 
	DxUint8_t					*SEP_dQ_ptr;
	DxUint16_t					 SEP_dQSize;
	DxUint16_t					 SEP_MaxdQSize = RSA_BufferMaxLength;//CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8; 
	DxUint8_t					*SEP_qInv_ptr;
	DxUint16_t					 SEP_qInvSize;
	DxUint16_t					 SEP_MaxqInvSize = RSA_BufferMaxLength;//CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8; 
	DxUint8_t					*SEP_Data_ptr;
	DxUint8_t					*SEP_ExpectedData_ptr;
	DxUint16_t					 SEP_MaxExpectedDataSize = RSA_BufferMaxLength;//CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8; 

	/* 1. extract parameters from the message */
	/******************************************/
	SEP_DataInSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_DATASIZE_IN_OFFSET];

	SEP_PubExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET];
	SEP_PrivExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET];
	/* modulus size */
	SEP_KeySize     = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_KEYSIZE_OFFSET];
	SEP_CRT_Mode    = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_CRTMODE_OFFSET];
	SEP_PSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PSIZE_OFFSET];
	SEP_QSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_QSIZE_OFFSET];
	SEP_dPSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dPSIZE_OFFSET];
	SEP_dQSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dQSIZE_OFFSET];
	SEP_qInvSize    = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_qInvSIZE_OFFSET];

	/* message length in words = length of the not buffer parameters */
	/* + all buffers parameters maximal lengthes                     */
	SEP_MsgLengthInWords = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS + 
		SEP_MaxDataInSize/4 + SEP_MaxPubExponentSize/4 + SEP_MaxPrivExponentSize/4 +
		SEP_MaxKeySize/4 + SEP_MaxPSize/4 + SEP_MaxQSize/4 + SEP_MaxdPSize/4 + 
		SEP_MaxdQSize/4 + SEP_MaxqInvSize/4 + SEP_MaxExpectedDataSize/4;

	SEP_PubExponent_ptr  = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PubExponent_OFFSET];

	SEP_PrivExponent_ptr = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PrivExponent_OFFSET];
	SEP_Modulus_ptr		 = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Modulus_OFFSET];
	SEP_P_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_P_OFFSET];
	SEP_Q_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Q_OFFSET];
	SEP_dP_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dP_OFFSET];
	SEP_dQ_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dQ_OFFSET];
	SEP_qInv_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_qInv_OFFSET];
	SEP_Data_ptr           = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_DataIn_OFFSET];
	SEP_ExpectedData_ptr   = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_DataExpected_OFFSET];


	/* 2. call QA SEP function */
	/***************************/
	SEP_Error = SEPQA_RSA_PRIME_SignVerify(	SEP_PubExponent_ptr,
											SEP_PubExponentSize,
											SEP_PrivExponent_ptr,
											SEP_Modulus_ptr,
											SEP_KeySize,
											SEP_CRT_Mode,
											SEP_P_ptr, 
											SEP_PSize,
											SEP_Q_ptr,
											SEP_QSize,
											SEP_dP_ptr, 
											SEP_dPSize,
											SEP_dQ_ptr,
											SEP_dQSize,
											SEP_qInv_ptr,
											SEP_qInvSize,
											SEP_Data_ptr,
											SEP_DataInSize,
											SEP_ExpectedData_ptr,
											FailureFunctionNameOut);

	if(SEP_Error != CRYS_OK)
	{
		/* output parameters are placed after msg RC */
		DX_VOS_FastMemCpy((DxUint8_t  *)&msg[DX_SEP_HOST_SEP_PROTOCOL_MSG_RC_OFFSET + 1], (DxUint8_t  *)&FailureFunctionNameOut[0], TST_FUNC_STRING_LEN);
		SEP_MsgLengthInWords = TST_FUNC_STRING_LEN/sizeof(DxUint32_t);
	}

	/* 3. end SEP message and return */
	/**********************/
	return DX_SEP_EndMessage(msg, SEP_Error, SEP_MsgLengthInWords*sizeof(DxUint32_t));
}

/*****************************************************************************
* Function Name:                                                           
*  DX_SEP_PARSER_RSA_PKCS1v15_EncDec
*  
* Inputs:
*  msg - message with all requested parameters.
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
* This function does the parsing for the external application that executes
* RSA PKCS1v15 encrypt and decrypt according to PKCS#1 Ver1.5
* It calls the SEP function with parameters parsed from input message
* Algorithm: 
* 1. extract parameters from the message 
* 2. call QA SEP function
* 3. end SEP message and return
******************************************************************************/                                
DxUint32_t DX_SEP_PARSER_RSA_PKCS1v15_EncDec(DxUint32_t* msg)
{

	/********* Variables Definitions ************/
	DxUint32_t					SEP_Error = DX_SUCCESS;
	/* input data size in bytes: */
	DxUint16_t					SEP_DataInSize = 0; 
	/* input data size word aligned */
	DxUint32_t					SEP_MaxDataInSize = RSA_BufferMaxLength; 
	/* message length in words */
	DxUint32_t					SEP_MsgLengthInWords;

	DxUint8_t	                *SEP_PubExponent_ptr;
	DxUint16_t					 SEP_PubExponentSize = 0;
	DxUint16_t					 SEP_MaxPubExponentSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_PrivExponent_ptr;
	DxUint16_t					 SEP_PrivExponentSize;
	DxUint16_t					 SEP_MaxPrivExponentSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_Modulus_ptr;
	/* modulus size */
	DxUint16_t					 SEP_KeySize;
	DxUint16_t					 SEP_MaxKeySize = RSA_BufferMaxLength; 
	DxUint8_t					 SEP_CRT_Mode;
	DxUint8_t					 SEP_KeyGenType;
	DxUint8_t					*SEP_P_ptr; 
	DxUint16_t					 SEP_PSize;
	DxUint16_t					 SEP_MaxPSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_Q_ptr;
	DxUint16_t					 SEP_QSize;
	DxUint16_t					 SEP_MaxQSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_dP_ptr;
	DxUint16_t					 SEP_dPSize;
	DxUint16_t					 SEP_MaxdPSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_dQ_ptr;
	DxUint16_t					 SEP_dQSize;
	DxUint16_t					 SEP_MaxdQSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_qInv_ptr;
	DxUint16_t					 SEP_qInvSize;
	DxUint16_t					 SEP_MaxqInvSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_Data_ptr;
	

	/* 1. extract parameters from the message */
	/******************************************/
	SEP_DataInSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_DATASIZE_IN_OFFSET];

	SEP_PubExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET];
	SEP_PrivExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET];
	/* modulus size */
	SEP_KeySize     = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_KEYSIZE_OFFSET];
	SEP_CRT_Mode    = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_CRTMODE_OFFSET];
	SEP_KeyGenType  = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_KEYGENTYPE_OFFSET];
	SEP_PSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PSIZE_OFFSET];
	SEP_QSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_QSIZE_OFFSET];
	SEP_dPSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dPSIZE_OFFSET];
	SEP_dQSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dQSIZE_OFFSET];
	SEP_qInvSize    = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_qInvSIZE_OFFSET];


	/* message length in words = length of the not buffer parameters */
	/* + all buffers parameters maximal lengthes                     */
	SEP_MsgLengthInWords = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS + 
		SEP_MaxDataInSize/4 + SEP_MaxPubExponentSize/4 + SEP_MaxPrivExponentSize/4 +
		SEP_MaxKeySize/4 + SEP_MaxPSize/4 + SEP_MaxQSize/4 + SEP_MaxdPSize/4 + 
		SEP_MaxdQSize/4 + SEP_MaxqInvSize/4 /*+ SEP_MaxExpectedDataSize/4*/;

	SEP_PubExponent_ptr  = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PubExponent_OFFSET];
	SEP_PrivExponent_ptr = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PrivExponent_OFFSET];
	SEP_Modulus_ptr		 = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Modulus_OFFSET];
	SEP_P_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_P_OFFSET];
	SEP_Q_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Q_OFFSET];
	SEP_dP_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dP_OFFSET];
	SEP_dQ_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dQ_OFFSET];
	SEP_qInv_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_qInv_OFFSET];
	SEP_Data_ptr           = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_DataIn_OFFSET];
	
	/* 2. call QA SEP function */
	/***************************/
	SEP_Error = SEPQA_RSA_PKCS1v15_EncDec (	SEP_PubExponent_ptr,
											SEP_PubExponentSize,
											SEP_PrivExponent_ptr,
											SEP_PrivExponentSize,
											SEP_Modulus_ptr,
											SEP_KeySize,
											SEP_KeyGenType,
											SEP_CRT_Mode,
											SEP_P_ptr, 
											SEP_PSize,
											SEP_Q_ptr,
											SEP_QSize,
											SEP_dP_ptr, 
											SEP_dPSize,
											SEP_dQ_ptr,
											SEP_dQSize,
											SEP_qInv_ptr,
											SEP_qInvSize,
											SEP_Data_ptr,
											SEP_DataInSize,
											FailureFunctionNameOut);

	if(SEP_Error != CRYS_OK)
	{
		/* output parameters are placed after msg RC */
		DX_VOS_FastMemCpy((DxUint8_t  *)&msg[DX_SEP_HOST_SEP_PROTOCOL_MSG_RC_OFFSET + 1], (DxUint8_t  *)&FailureFunctionNameOut[0], TST_FUNC_STRING_LEN);
		SEP_MsgLengthInWords = TST_FUNC_STRING_LEN/sizeof(DxUint32_t);
	}

	/* 3. end SEP message and return */
	/**********************/
	return DX_SEP_EndMessage(msg, SEP_Error, SEP_MsgLengthInWords*sizeof(DxUint32_t));
}


/*****************************************************************************
* Function Name:                                                           
*  DX_SEP_PARSER_RSA_PKCS1v15_NonIntegrate_SignVerify
*  
* Inputs:
*  msg - message with all requested parameters.
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
* This function does the parsing for the external application that executes
* acceptance non integrated test for RSA PKCS1v15 Sign & Verify
* It calls the SEP function with parameters parsed from input message
* Algorithm: 
* 1. extract parameters from the message 
* 2. call QA SEP function
* 3. end SEP message and return
******************************************************************************/                                
DxUint32_t DX_SEP_PARSER_RSA_PKCS1v15_NonIntegrate_SignVerify(DxUint32_t* msg)
{

	/********* Variables Definitions ************/
	DxUint32_t					SEP_Error = DX_SUCCESS;
	/* input data size in bytes: */
	DxUint16_t					SEP_DataInSize = 0; 
	/* input data size word aligned */
	DxUint32_t					SEP_MaxDataInSize = 0;
	/* message length in words */
	DxUint32_t					SEP_MsgLengthInWords;

	DxUint8_t	                *SEP_PubExponent_ptr;
	DxUint16_t					 SEP_PubExponentSize = 0;
	DxUint16_t					 SEP_MaxPubExponentSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_PrivExponent_ptr;
	DxUint16_t					 SEP_PrivExponentSize;
	DxUint16_t					 SEP_MaxPrivExponentSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Modulus_ptr;
	DxUint8_t					 SEP_KeyGenType;
	CRYS_RSA_HASH_OpMode_t       SEP_HashMode;
	/* modulus size */
	DxUint16_t					 SEP_KeySize;
	DxUint16_t					 SEP_MaxKeySize = RSA_BufferMaxLength;
	DxUint8_t					 SEP_CRT_Mode;
	DxUint8_t					*SEP_P_ptr; 
	DxUint16_t					 SEP_PSize;
	DxUint16_t					 SEP_MaxPSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Q_ptr;
	DxUint16_t					 SEP_QSize;
	DxUint16_t					 SEP_MaxQSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_dP_ptr;
	DxUint16_t					 SEP_dPSize;
	DxUint16_t					 SEP_MaxdPSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_dQ_ptr;
	DxUint16_t					 SEP_dQSize;
	DxUint16_t					 SEP_MaxdQSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_qInv_ptr;
	DxUint16_t					 SEP_qInvSize;
	DxUint16_t					 SEP_MaxqInvSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_Data_ptr;
	

	/* 1. extract parameters from the message */
	/******************************************/
	SEP_DataInSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_DATASIZE_IN_OFFSET];

	SEP_PubExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET];
	SEP_PrivExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET];
	/* modulus size */
	SEP_KeySize     = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_KEYSIZE_OFFSET];
	SEP_CRT_Mode    = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_CRTMODE_OFFSET];
	SEP_PSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PSIZE_OFFSET];
	SEP_QSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_QSIZE_OFFSET];
	SEP_dPSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dPSIZE_OFFSET];
	SEP_dQSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dQSIZE_OFFSET];
	SEP_qInvSize    = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_qInvSIZE_OFFSET];
	SEP_KeyGenType  = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_KEYGENTYPE_OFFSET];
	SEP_HashMode    = (CRYS_RSA_HASH_OpMode_t)msg[DX_EXT_SEP_HOST_RSA_HASHMODE_OFFSET];

	SEP_MaxDataInSize = ((SEP_DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);


	/* message length in words = length of the not buffer parameters */
	/* + all buffers parameters maximal lengthes                     */
	SEP_MsgLengthInWords = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS + 
		SEP_MaxDataInSize/4 + SEP_MaxPubExponentSize/4 + SEP_MaxPrivExponentSize/4 +
		SEP_MaxKeySize/4 + SEP_MaxPSize/4 + SEP_MaxQSize/4 + SEP_MaxdPSize/4 + 
		SEP_MaxdQSize/4 + SEP_MaxqInvSize/4 /*+ SEP_MaxExpectedDataSize/4*/;

	SEP_PubExponent_ptr  = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PubExponent_OFFSET];
	SEP_PrivExponent_ptr = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PrivExponent_OFFSET];
	SEP_Modulus_ptr		 = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Modulus_OFFSET];
	SEP_P_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_P_OFFSET];
	SEP_Q_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Q_OFFSET];
	SEP_dP_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dP_OFFSET];
	SEP_dQ_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dQ_OFFSET];
	SEP_qInv_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_qInv_OFFSET];
	SEP_Data_ptr           = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_DataIn_OFFSET];


	/* 2. call QA SEP function */
	/***************************/
	SEP_Error = SEPQA_RSA_PKCS1v15_NonIntegrate_SignVerify (SEP_PubExponent_ptr,
															SEP_PubExponentSize,
															SEP_PrivExponent_ptr,
															SEP_PrivExponentSize,
															SEP_Modulus_ptr,
															SEP_KeySize,
															SEP_CRT_Mode,
															SEP_KeyGenType,
															SEP_HashMode,
															SEP_P_ptr, 
															SEP_PSize,
															SEP_Q_ptr,
															SEP_QSize,
															SEP_dP_ptr, 
															SEP_dPSize,
															SEP_dQ_ptr,
															SEP_dQSize,
															SEP_qInv_ptr,
															SEP_qInvSize,
															SEP_Data_ptr,
															SEP_DataInSize,
															FailureFunctionNameOut);

	if(SEP_Error != CRYS_OK)
	{
		/* output parameters are placed after msg RC */
		DX_VOS_FastMemCpy((DxUint8_t  *)&msg[DX_SEP_HOST_SEP_PROTOCOL_MSG_RC_OFFSET + 1], (DxUint8_t  *)&FailureFunctionNameOut[0], TST_FUNC_STRING_LEN);
		SEP_MsgLengthInWords = TST_FUNC_STRING_LEN/sizeof(DxUint32_t);
	}

	/* 3. end SEP message and return */
	/**********************/
	return DX_SEP_EndMessage(msg, SEP_Error, SEP_MsgLengthInWords*sizeof(DxUint32_t));
}

/*****************************************************************************
* Function Name:                                                           
*  DX_SEP_PARSER_RSA_PKCS1v15_Integrate_SignVerify
*  
* Inputs:
*  msg - message with all requested parameters.
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
* This function does the parsing for the external application that executes
* acceptance integrated test for RSA PKCS1v15 Sign & Verify
* It calls the SEP function with parameters parsed from input message
* Algorithm: 
* 1. extract parameters from the message 
* 2. call QA SEP function
* 3. end SEP message and return
******************************************************************************/                                
DxUint32_t DX_SEP_PARSER_RSA_PKCS1v15_Integrate_SignVerify(DxUint32_t* msg)
{

	/********* Variables Definitions ************/
	DxUint32_t					SEP_Error = DX_SUCCESS;
	/* input data size in bytes: */
	DxUint16_t					SEP_DataInSize = 0; 
	/* input data size word aligned */
	DxUint32_t					SEP_MaxDataInSize = 0;
	/* message length in words */
	DxUint32_t					SEP_MsgLengthInWords;

	DxUint8_t	                *SEP_PubExponent_ptr;
	DxUint16_t					 SEP_PubExponentSize = 0;
	DxUint16_t					 SEP_MaxPubExponentSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_PrivExponent_ptr;
	DxUint16_t					 SEP_PrivExponentSize;
	DxUint16_t					 SEP_MaxPrivExponentSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Modulus_ptr;
	DxUint8_t					 SEP_KeyGenType;
	CRYS_RSA_HASH_OpMode_t       SEP_HashMode;
	/* modulus size */
	DxUint16_t					 SEP_KeySize;
	DxUint16_t					 SEP_MaxKeySize = RSA_BufferMaxLength;
	DxUint8_t					 SEP_CRT_Mode;
	DxUint8_t					*SEP_P_ptr; 
	DxUint16_t					 SEP_PSize;
	DxUint16_t					 SEP_MaxPSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Q_ptr;
	DxUint16_t					 SEP_QSize;
	DxUint16_t					 SEP_MaxQSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_dP_ptr;
	DxUint16_t					 SEP_dPSize;
	DxUint16_t					 SEP_MaxdPSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_dQ_ptr;
	DxUint16_t					 SEP_dQSize;
	DxUint16_t					 SEP_MaxdQSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_qInv_ptr;
	DxUint16_t					 SEP_qInvSize;
	DxUint16_t					 SEP_MaxqInvSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Data_ptr;
	
	/* 1. extract parameters from the message */
	/******************************************/
	SEP_DataInSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_DATASIZE_IN_OFFSET];

	SEP_PubExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET];
	SEP_PrivExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET];
	/* modulus size */
	SEP_KeySize     = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_KEYSIZE_OFFSET];
	SEP_CRT_Mode    = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_CRTMODE_OFFSET];
	SEP_PSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PSIZE_OFFSET];
	SEP_QSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_QSIZE_OFFSET];
	SEP_dPSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dPSIZE_OFFSET];
	SEP_dQSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dQSIZE_OFFSET];
	SEP_qInvSize    = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_qInvSIZE_OFFSET];
	SEP_KeyGenType  = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_KEYGENTYPE_OFFSET];
	SEP_HashMode    = (CRYS_RSA_HASH_OpMode_t)msg[DX_EXT_SEP_HOST_RSA_HASHMODE_OFFSET];

	SEP_MaxDataInSize = ((SEP_DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);

	/* message length in words = length of the not buffer parameters */
	/* + all buffers parameters maximal lengthes                     */
	SEP_MsgLengthInWords = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS + 
		SEP_MaxDataInSize/4 + SEP_MaxPubExponentSize/4 + SEP_MaxPrivExponentSize/4 +
		SEP_MaxKeySize/4 + SEP_MaxPSize/4 + SEP_MaxQSize/4 + SEP_MaxdPSize/4 + 
		SEP_MaxdQSize/4 + SEP_MaxqInvSize/4 /*+ SEP_MaxExpectedDataSize/4*/;

	SEP_PubExponent_ptr  = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PubExponent_OFFSET];
	SEP_PrivExponent_ptr = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PrivExponent_OFFSET];
	SEP_Modulus_ptr		 = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Modulus_OFFSET];
	SEP_P_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_P_OFFSET];
	SEP_Q_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Q_OFFSET];
	SEP_dP_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dP_OFFSET];
	SEP_dQ_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dQ_OFFSET];
	SEP_qInv_ptr         =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_qInv_OFFSET];
	SEP_Data_ptr         = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_DataIn_OFFSET];
	

	/* 2. call QA SEP function */
	/***************************/
	SEP_Error = SEPQA_RSA_PKCS1v15_Integrate_SignVerify(SEP_PubExponent_ptr,
														SEP_PubExponentSize,
														SEP_PrivExponent_ptr,
														SEP_PrivExponentSize,
														SEP_Modulus_ptr,
														SEP_KeySize,
														SEP_CRT_Mode,
														SEP_KeyGenType,
														SEP_HashMode,
														SEP_P_ptr, 
														SEP_PSize,
														SEP_Q_ptr,
														SEP_QSize,
														SEP_dP_ptr, 
														SEP_dPSize,
														SEP_dQ_ptr,
														SEP_dQSize,
														SEP_qInv_ptr,
														SEP_qInvSize,
														SEP_Data_ptr,
														SEP_DataInSize,
														FailureFunctionNameOut);

	if(SEP_Error != CRYS_OK)
	{
		/* output parameters are placed after msg RC */
		DX_VOS_FastMemCpy((DxUint8_t  *)&msg[DX_SEP_HOST_SEP_PROTOCOL_MSG_RC_OFFSET + 1], (DxUint8_t  *)&FailureFunctionNameOut[0], TST_FUNC_STRING_LEN);
		SEP_MsgLengthInWords = TST_FUNC_STRING_LEN/sizeof(DxUint32_t);
	}

	/* 3. end SEP message and return */
	/**********************/
	return DX_SEP_EndMessage(msg, SEP_Error, SEP_MsgLengthInWords*sizeof(DxUint32_t));
}

/*****************************************************************************
* Function Name:                                                           
*  DX_SEP_PARSER_RSA_OAEP_EncDec
*  
* Inputs:
*  msg - message with all requested parameters.
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
* This function does the parsing for the external application that executes
* acceptance test for RSA OAEP encrypt and decrypt
* according to PKCS#1 Ver2.1
* It calls the SEP function with parameters parsed from input message
* Algorithm: 
* 1. extract parameters from the message 
* 2. call QA SEP function
* 3. end SEP message and return
******************************************************************************/                                

DxUint32_t DX_SEP_PARSER_RSA_OAEP_EncDec(DxUint32_t* msg)
{

	/********* Variables Definitions ************/
	DxUint32_t					SEP_Error = DX_SUCCESS;
	/* input data size in bytes: */
	DxUint16_t					SEP_DataInSize = 0; 
	/* input data size word aligned */
	DxUint32_t					SEP_MaxDataInSize = RSA_BufferMaxLength; 
	/* message length in words */
	DxUint32_t					SEP_MsgLengthInWords;

	DxUint8_t	                *SEP_PubExponent_ptr;
	DxUint16_t					 SEP_PubExponentSize = 0;
	DxUint16_t					 SEP_MaxPubExponentSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_PrivExponent_ptr;
	DxUint16_t					 SEP_PrivExponentSize;
	DxUint16_t					 SEP_MaxPrivExponentSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_Modulus_ptr;
	DxUint8_t					 SEP_KeyGenType;
	CRYS_RSA_HASH_OpMode_t       SEP_HashMode;
	/* modulus size */
	DxUint16_t					 SEP_KeySize;
	DxUint16_t					 SEP_MaxKeySize = RSA_BufferMaxLength; 
	DxUint8_t					 SEP_CRT_Mode;
	DxUint8_t					*SEP_P_ptr; 
	DxUint16_t					 SEP_PSize;
	DxUint16_t					 SEP_MaxPSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_Q_ptr;
	DxUint16_t					 SEP_QSize;
	DxUint16_t					 SEP_MaxQSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_dP_ptr;
	DxUint16_t					 SEP_dPSize;
	DxUint16_t					 SEP_MaxdPSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_dQ_ptr;
	DxUint16_t					 SEP_dQSize;
	DxUint16_t					 SEP_MaxdQSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_qInv_ptr;
	DxUint16_t					 SEP_qInvSize;
	DxUint16_t					 SEP_MaxqInvSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_Data_ptr;
	

	/* 1. extract parameters from the message */
	/******************************************/
	SEP_DataInSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_DATASIZE_IN_OFFSET];

	SEP_PubExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET];
	SEP_PrivExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET];
	/* modulus size */
	SEP_KeySize     = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_KEYSIZE_OFFSET];
	SEP_CRT_Mode    = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_CRTMODE_OFFSET];
	SEP_PSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PSIZE_OFFSET];
	SEP_QSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_QSIZE_OFFSET];
	SEP_dPSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dPSIZE_OFFSET];
	SEP_dQSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dQSIZE_OFFSET];
	SEP_qInvSize    = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_qInvSIZE_OFFSET];
	SEP_KeyGenType  = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_KEYGENTYPE_OFFSET];
	SEP_HashMode    = (CRYS_RSA_HASH_OpMode_t)msg[DX_EXT_SEP_HOST_RSA_HASHMODE_OFFSET];


	/* message length in words = length of the not buffer parameters */
	/* + all buffers parameters maximal lengthes                     */
	SEP_MsgLengthInWords = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS + 
		SEP_MaxDataInSize/4 + SEP_MaxPubExponentSize/4 + SEP_MaxPrivExponentSize/4 +
		SEP_MaxKeySize/4 + SEP_MaxPSize/4 + SEP_MaxQSize/4 + SEP_MaxdPSize/4 + 
		SEP_MaxdQSize/4 + SEP_MaxqInvSize/4 /*+ SEP_MaxExpectedDataSize/4*/;

	SEP_PubExponent_ptr  = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PubExponent_OFFSET];
	SEP_PrivExponent_ptr = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PrivExponent_OFFSET];
	SEP_Modulus_ptr		 = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Modulus_OFFSET];
	SEP_P_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_P_OFFSET];
	SEP_Q_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Q_OFFSET];
	SEP_dP_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dP_OFFSET];
	SEP_dQ_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dQ_OFFSET];
	SEP_qInv_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_qInv_OFFSET];
	SEP_Data_ptr           = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_DataIn_OFFSET];
	


	/* 2. call QA SEP function */
	/***************************/
	SEP_Error = SEPQA_RSA_OAEP_EncDec (	SEP_PubExponent_ptr,
										SEP_PubExponentSize,
										SEP_PrivExponent_ptr,
										SEP_PrivExponentSize,
										SEP_Modulus_ptr,
										SEP_KeySize,
										SEP_CRT_Mode,
										SEP_KeyGenType,
										SEP_HashMode,
										SEP_P_ptr, 
										SEP_PSize,
										SEP_Q_ptr,
										SEP_QSize,
										SEP_dP_ptr, 
										SEP_dPSize,
										SEP_dQ_ptr,
										SEP_dQSize,
										SEP_qInv_ptr,
										SEP_qInvSize,
										SEP_Data_ptr,
										SEP_DataInSize,
										FailureFunctionNameOut);

	if(SEP_Error != CRYS_OK)
	{
		/* output parameters are placed after msg RC */
		DX_VOS_FastMemCpy((DxUint8_t  *)&msg[DX_SEP_HOST_SEP_PROTOCOL_MSG_RC_OFFSET + 1], (DxUint8_t  *)&FailureFunctionNameOut[0], TST_FUNC_STRING_LEN);
		SEP_MsgLengthInWords = TST_FUNC_STRING_LEN/sizeof(DxUint32_t);
	}

	/* 3. end SEP message and return */
	/**********************/
	return DX_SEP_EndMessage(msg, SEP_Error, SEP_MsgLengthInWords*sizeof(DxUint32_t));
}

/*****************************************************************************
* Function Name:                                                           
*  DX_SEP_PARSER_RSA_PSS_NonIntegrate_SignVerify
*  
* Inputs:
*  msg - message with all requested parameters.
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
* This function does the parsing for the external application that executes acceptance test 
* for RSA PSS PKCS#1v2.1 non integrated Sign & Verify
* It calls the SEP function with parameters parsed from input message
* Algorithm: 
* 1. extract parameters from the message 
* 2. call QA SEP function
* 3. end SEP message and return
******************************************************************************/                                
DxUint32_t DX_SEP_PARSER_RSA_PSS_NonIntegrate_SignVerify(DxUint32_t* msg)
{

	/********* Variables Definitions ************/
	DxUint32_t					SEP_Error = DX_SUCCESS;
	/* input data size in bytes: */
	DxUint16_t					SEP_DataInSize = 0; 
	/* input data size word aligned */
	DxUint32_t					SEP_MaxDataInSize = RSA_BufferMaxLength; 
	/* message length in words */
	DxUint32_t					SEP_MsgLengthInWords;

	DxUint8_t	                *SEP_PubExponent_ptr;
	DxUint16_t					 SEP_PubExponentSize = 0;
	DxUint16_t					 SEP_MaxPubExponentSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_PrivExponent_ptr;
	DxUint16_t					 SEP_PrivExponentSize;
	DxUint16_t					 SEP_MaxPrivExponentSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Modulus_ptr;
	DxUint8_t					 SEP_KeyGenType;
	CRYS_RSA_HASH_OpMode_t       SEP_HashMode;
	DxUint16_t		             SEP_SaltLength;
	/* modulus size */
	DxUint16_t					 SEP_KeySize;
	DxUint16_t					 SEP_MaxKeySize = RSA_BufferMaxLength;
	DxUint8_t					 SEP_CRT_Mode;
	DxUint8_t					*SEP_P_ptr; 
	DxUint16_t					 SEP_PSize;
	DxUint16_t					 SEP_MaxPSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Q_ptr;
	DxUint16_t					 SEP_QSize;
	DxUint16_t					 SEP_MaxQSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_dP_ptr;
	DxUint16_t					 SEP_dPSize;
	DxUint16_t					 SEP_MaxdPSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_dQ_ptr;
	DxUint16_t					 SEP_dQSize;
	DxUint16_t					 SEP_MaxdQSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_qInv_ptr;
	DxUint16_t					 SEP_qInvSize;
	DxUint16_t					 SEP_MaxqInvSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Data_ptr;
	

	/* 1. extract parameters from the message */
	/******************************************/
	SEP_DataInSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_DATASIZE_IN_OFFSET];

	SEP_PubExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET];
	SEP_PrivExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET];
	/* modulus size */
	SEP_KeySize     = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_KEYSIZE_OFFSET];
	SEP_CRT_Mode    = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_CRTMODE_OFFSET];
	SEP_PSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PSIZE_OFFSET];
	SEP_QSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_QSIZE_OFFSET];
	SEP_dPSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dPSIZE_OFFSET];
	SEP_dQSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dQSIZE_OFFSET];
	SEP_qInvSize    = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_qInvSIZE_OFFSET];
	SEP_KeyGenType  = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_KEYGENTYPE_OFFSET];
	SEP_HashMode    = (CRYS_RSA_HASH_OpMode_t)msg[DX_EXT_SEP_HOST_RSA_HASHMODE_OFFSET];
	SEP_SaltLength  = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_SALTLENGTH_OFFSET];

	SEP_MaxDataInSize = ((SEP_DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);


	/* message length in words = length of the not buffer parameters */
	/* + all buffers parameters maximal lengthes                     */
	SEP_MsgLengthInWords = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS + 
		SEP_MaxDataInSize/4 + SEP_MaxPubExponentSize/4 + SEP_MaxPrivExponentSize/4 +
		SEP_MaxKeySize/4 + SEP_MaxPSize/4 + SEP_MaxQSize/4 + SEP_MaxdPSize/4 + 
		SEP_MaxdQSize/4 + SEP_MaxqInvSize/4 /*+ SEP_MaxExpectedDataSize/4*/;

	SEP_PubExponent_ptr  = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PubExponent_OFFSET];
	SEP_PrivExponent_ptr = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PrivExponent_OFFSET];
	SEP_Modulus_ptr		 = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Modulus_OFFSET];
	SEP_P_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_P_OFFSET];
	SEP_Q_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Q_OFFSET];
	SEP_dP_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dP_OFFSET];
	SEP_dQ_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dQ_OFFSET];
	SEP_qInv_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_qInv_OFFSET];
	SEP_Data_ptr           = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_DataIn_OFFSET];
	

	/* 2. call QA SEP function */
	/***************************/
	SEP_Error = SEPQA_RSA_PSS_NonIntegrate_SignVerify (	SEP_PubExponent_ptr,
														SEP_PubExponentSize,
														SEP_PrivExponent_ptr,
														SEP_PrivExponentSize,
														SEP_Modulus_ptr,
														SEP_KeySize,
														SEP_CRT_Mode,
														SEP_KeyGenType,
														SEP_HashMode,
														SEP_SaltLength,
														SEP_P_ptr, 
														SEP_PSize,
														SEP_Q_ptr,
														SEP_QSize,
														SEP_dP_ptr, 
														SEP_dPSize,
														SEP_dQ_ptr,
														SEP_dQSize,
														SEP_qInv_ptr,
														SEP_qInvSize,
														SEP_Data_ptr,
														SEP_DataInSize,
														FailureFunctionNameOut);

	if(SEP_Error != CRYS_OK)
	{
		/* output parameters are placed after msg RC */
		DX_VOS_FastMemCpy((DxUint8_t  *)&msg[DX_SEP_HOST_SEP_PROTOCOL_MSG_RC_OFFSET + 1], (DxUint8_t  *)&FailureFunctionNameOut[0], TST_FUNC_STRING_LEN);
		SEP_MsgLengthInWords = TST_FUNC_STRING_LEN/sizeof(DxUint32_t);
	}

	/* 3. end SEP message and return */
	/**********************/
	return DX_SEP_EndMessage(msg, SEP_Error, SEP_MsgLengthInWords*sizeof(DxUint32_t));
}


/*****************************************************************************
* Function Name:                                                           
*  DX_SEP_PARSER_RSA_PSS_Integrate_SignVerify
*  
* Inputs:
*  msg - message with all requested parameters.
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
* This function does the parsing for the external application that executes acceptance test 
* for RSA PSS PKCS#1v2.1 integrated sign & Verify
* It calls the SEP function with parameters parsed from input message
* Algorithm: 
* 1. extract parameters from the message 
* 2. call QA SEP function
* 3. end SEP message and return
******************************************************************************/                                
DxUint32_t DX_SEP_PARSER_RSA_PSS_Integrate_SignVerify(DxUint32_t* msg)
{

	/********* Variables Definitions ************/
	DxUint32_t					SEP_Error = DX_SUCCESS;
	/* input data size in bytes: */
	DxUint16_t					SEP_DataInSize = 0; 
	/* input data size word aligned */
	DxUint32_t					SEP_MaxDataInSize =  RSA_BufferMaxLength; 
	/* message length in words */
	DxUint32_t					SEP_MsgLengthInWords;

	DxUint8_t	                *SEP_PubExponent_ptr;
	DxUint16_t					 SEP_PubExponentSize = 0;
	DxUint16_t					 SEP_MaxPubExponentSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_PrivExponent_ptr;
	DxUint16_t					 SEP_PrivExponentSize;
	DxUint16_t					 SEP_MaxPrivExponentSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Modulus_ptr;
	DxUint8_t					 SEP_KeyGenType;
	CRYS_RSA_HASH_OpMode_t       SEP_HashMode;
	DxUint16_t		             SEP_SaltLength;
	/* modulus size */
	DxUint16_t					 SEP_KeySize;
	DxUint16_t					 SEP_MaxKeySize = RSA_BufferMaxLength;
	DxUint8_t					 SEP_CRT_Mode;
	DxUint8_t					*SEP_P_ptr; 
	DxUint16_t					 SEP_PSize;
	DxUint16_t					 SEP_MaxPSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Q_ptr;
	DxUint16_t					 SEP_QSize;
	DxUint16_t					 SEP_MaxQSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_dP_ptr;
	DxUint16_t					 SEP_dPSize;
	DxUint16_t					 SEP_MaxdPSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_dQ_ptr;
	DxUint16_t					 SEP_dQSize;
	DxUint16_t					 SEP_MaxdQSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_qInv_ptr;
	DxUint16_t					 SEP_qInvSize;
	DxUint16_t					 SEP_MaxqInvSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Data_ptr;
	

	/* 1. extract parameters from the message */
	/******************************************/
	SEP_DataInSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_DATASIZE_IN_OFFSET];

	SEP_PubExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET];
	SEP_PrivExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET];
	/* modulus size */
	SEP_KeySize     = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_KEYSIZE_OFFSET];
	SEP_CRT_Mode    = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_CRTMODE_OFFSET];
	SEP_PSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PSIZE_OFFSET];
	SEP_QSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_QSIZE_OFFSET];
	SEP_dPSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dPSIZE_OFFSET];
	SEP_dQSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dQSIZE_OFFSET];
	SEP_qInvSize    = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_qInvSIZE_OFFSET];
	SEP_KeyGenType  = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_KEYGENTYPE_OFFSET];
	SEP_HashMode    = (CRYS_RSA_HASH_OpMode_t)msg[DX_EXT_SEP_HOST_RSA_HASHMODE_OFFSET];
	SEP_SaltLength  = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_SALTLENGTH_OFFSET];

	SEP_MaxDataInSize = ((SEP_DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);


	/* message length in words = length of the not buffer parameters */
	/* + all buffers parameters maximal lengthes                     */
	SEP_MsgLengthInWords = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS + 
		SEP_MaxDataInSize/4 + SEP_MaxPubExponentSize/4 + SEP_MaxPrivExponentSize/4 +
		SEP_MaxKeySize/4 + SEP_MaxPSize/4 + SEP_MaxQSize/4 + SEP_MaxdPSize/4 + 
		SEP_MaxdQSize/4 + SEP_MaxqInvSize/4 /*+ SEP_MaxExpectedDataSize/4*/;

	SEP_PubExponent_ptr  = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PubExponent_OFFSET];
	SEP_PrivExponent_ptr = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PrivExponent_OFFSET];
	SEP_Modulus_ptr		 = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Modulus_OFFSET];
	SEP_P_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_P_OFFSET];
	SEP_Q_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Q_OFFSET];
	SEP_dP_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dP_OFFSET];
	SEP_dQ_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dQ_OFFSET];
	SEP_qInv_ptr         =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_qInv_OFFSET];
	SEP_Data_ptr         = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_DataIn_OFFSET];
	

	/* 2. call QA SEP function */
	/***************************/
	SEP_Error = SEPQA_RSA_PSS_Integrate_SignVerify (SEP_PubExponent_ptr,
													SEP_PubExponentSize,
													SEP_PrivExponent_ptr,
													SEP_PrivExponentSize,
													SEP_Modulus_ptr,
													SEP_KeySize,
													SEP_CRT_Mode,
													SEP_KeyGenType,
													SEP_HashMode,
													SEP_SaltLength,
													SEP_P_ptr, 
													SEP_PSize,
													SEP_Q_ptr,
													SEP_QSize,
													SEP_dP_ptr, 
													SEP_dPSize,
													SEP_dQ_ptr,
													SEP_dQSize,
													SEP_qInv_ptr,
													SEP_qInvSize,
													SEP_Data_ptr,
													SEP_DataInSize,
													FailureFunctionNameOut);

	if(SEP_Error != CRYS_OK)
	{
		/* output parameters are placed after msg RC */
		DX_VOS_FastMemCpy((DxUint8_t  *)&msg[DX_SEP_HOST_SEP_PROTOCOL_MSG_RC_OFFSET + 1], (DxUint8_t  *)&FailureFunctionNameOut[0], TST_FUNC_STRING_LEN);
		SEP_MsgLengthInWords = TST_FUNC_STRING_LEN/sizeof(DxUint32_t);
	}

	/* 3. end SEP message and return */
	/**********************/
	return DX_SEP_EndMessage(msg, SEP_Error, SEP_MsgLengthInWords*sizeof(DxUint32_t));
}


/*****************************************************************************
* Function Name:                                                           
*  DX_SEP_PARSER_RSA_PKCS1v15_ExternalHash_SignVerify
*  
* Inputs:
*  msg - message with all requested parameters.
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
* This function does the parsing for the external application that executes
* acceptance non integrated test for RSA PKCS1v15 Sign & Verify
* It calls the SEP function with parameters parsed from input message
* Algorithm: 
* 1. extract parameters from the message 
* 2. call QA SEP function
* 3. end SEP message and return
******************************************************************************/                                
DxUint32_t DX_SEP_PARSER_RSA_PKCS1v15_ExternalHash_SignVerify(DxUint32_t* msg)
{

	/********* Variables Definitions ************/
	DxUint32_t					SEP_Error = DX_SUCCESS;
	/* input data size in bytes: */
	DxUint16_t					SEP_DataInSize = 0; 
	/* input data size word aligned */
	DxUint32_t					SEP_MaxDataInSize = 0;
	/* message length in words */
	DxUint32_t					SEP_MsgLengthInWords;

	DxUint8_t	                *SEP_PubExponent_ptr;
	DxUint16_t					 SEP_PubExponentSize = 0;
	DxUint16_t					 SEP_MaxPubExponentSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_PrivExponent_ptr;
	DxUint16_t					 SEP_PrivExponentSize;
	DxUint16_t					 SEP_MaxPrivExponentSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Modulus_ptr;
	DxUint8_t					 SEP_KeyGenType;
	CRYS_RSA_HASH_OpMode_t       SEP_HashMode;
	/* modulus size */
	DxUint16_t					 SEP_KeySize;
	DxUint16_t					 SEP_MaxKeySize = RSA_BufferMaxLength;
	DxUint8_t					 SEP_CRT_Mode;
	DxUint8_t					*SEP_P_ptr; 
	DxUint16_t					 SEP_PSize;
	DxUint16_t					 SEP_MaxPSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Q_ptr;
	DxUint16_t					 SEP_QSize;
	DxUint16_t					 SEP_MaxQSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_dP_ptr;
	DxUint16_t					 SEP_dPSize;
	DxUint16_t					 SEP_MaxdPSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_dQ_ptr;
	DxUint16_t					 SEP_dQSize;
	DxUint16_t					 SEP_MaxdQSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_qInv_ptr;
	DxUint16_t					 SEP_qInvSize;
	DxUint16_t					 SEP_MaxqInvSize = RSA_BufferMaxLength; 
	DxUint8_t					*SEP_Data_ptr;


	/* 1. extract parameters from the message */
	/******************************************/
	SEP_DataInSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_DATASIZE_IN_OFFSET];

	SEP_PubExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET];
	SEP_PrivExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET];
	/* modulus size */
	SEP_KeySize     = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_KEYSIZE_OFFSET];
	SEP_CRT_Mode    = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_CRTMODE_OFFSET];
	SEP_PSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PSIZE_OFFSET];
	SEP_QSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_QSIZE_OFFSET];
	SEP_dPSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dPSIZE_OFFSET];
	SEP_dQSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dQSIZE_OFFSET];
	SEP_qInvSize    = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_qInvSIZE_OFFSET];
	SEP_KeyGenType  = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_KEYGENTYPE_OFFSET];
	SEP_HashMode    = (CRYS_RSA_HASH_OpMode_t)msg[DX_EXT_SEP_HOST_RSA_HASHMODE_OFFSET];

	SEP_MaxDataInSize = ((SEP_DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);


	/* message length in words = length of the not buffer parameters */
	/* + all buffers parameters maximal lengthes                     */
	SEP_MsgLengthInWords = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS + 
		SEP_MaxDataInSize/4 + SEP_MaxPubExponentSize/4 + SEP_MaxPrivExponentSize/4 +
		SEP_MaxKeySize/4 + SEP_MaxPSize/4 + SEP_MaxQSize/4 + SEP_MaxdPSize/4 + 
		SEP_MaxdQSize/4 + SEP_MaxqInvSize/4 /*+ SEP_MaxExpectedDataSize/4*/;

	SEP_PubExponent_ptr  = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PubExponent_OFFSET];
	SEP_PrivExponent_ptr = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PrivExponent_OFFSET];
	SEP_Modulus_ptr		 = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Modulus_OFFSET];
	SEP_P_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_P_OFFSET];
	SEP_Q_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Q_OFFSET];
	SEP_dP_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dP_OFFSET];
	SEP_dQ_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dQ_OFFSET];
	SEP_qInv_ptr         =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_qInv_OFFSET];
	SEP_Data_ptr         = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_DataIn_OFFSET];


	/* 2. call QA SEP function */
	/***************************/
	SEP_Error = SEPQA_RSA_PKCS1v15_ExternalHash_SignVerify(	SEP_PubExponent_ptr,
															SEP_PubExponentSize,
															SEP_PrivExponent_ptr,
															SEP_Modulus_ptr,
															SEP_KeySize,
															SEP_CRT_Mode,
															SEP_KeyGenType,
															SEP_HashMode,
															SEP_P_ptr, 
															SEP_PSize,
															SEP_Q_ptr,
															SEP_QSize,
															SEP_dP_ptr, 
															SEP_dPSize,
															SEP_dQ_ptr,
															SEP_dQSize,
															SEP_qInv_ptr,
															SEP_qInvSize,
															SEP_Data_ptr,
															SEP_DataInSize,
															FailureFunctionNameOut);

	if(SEP_Error != CRYS_OK)
	{
		/* output parameters are placed after msg RC */
		DX_VOS_FastMemCpy((DxUint8_t  *)&msg[DX_SEP_HOST_SEP_PROTOCOL_MSG_RC_OFFSET + 1], (DxUint8_t  *)&FailureFunctionNameOut[0], TST_FUNC_STRING_LEN);
		SEP_MsgLengthInWords = TST_FUNC_STRING_LEN/sizeof(DxUint32_t);
	}

	/* 3. end SEP message and return */
	/**********************/
	return DX_SEP_EndMessage(msg, SEP_Error, SEP_MsgLengthInWords*sizeof(DxUint32_t));
}


/*****************************************************************************
* Function Name:                                                           
*  DX_SEP_PARSER_RSA_PSS_ExternalHash_SignVerify
*  
* Inputs:
*  msg - message with all requested parameters.
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
* This function does the parsing for the external application that executes acceptance test 
* for RSA PSS PKCS#1v2.1 integrated sign & Verify
* It calls the SEP function with parameters parsed from input message
* Algorithm: 
* 1. extract parameters from the message 
* 2. call QA SEP function
* 3. end SEP message and return
******************************************************************************/                                
DxUint32_t DX_SEP_PARSER_RSA_PSS_ExternalHash_SignVerify(DxUint32_t* msg)
{

	/********* Variables Definitions ************/
	DxUint32_t					SEP_Error = DX_SUCCESS;
	/* input data size in bytes: */
	DxUint16_t					SEP_DataInSize = 0; 
	/* input data size word aligned */
	DxUint32_t					SEP_MaxDataInSize =  RSA_BufferMaxLength; 
	/* message length in words */
	DxUint32_t					SEP_MsgLengthInWords;
	DxUint8_t	                *SEP_PubExponent_ptr;
	DxUint16_t					 SEP_PubExponentSize = 0;
	DxUint16_t					 SEP_MaxPubExponentSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_PrivExponent_ptr;
	DxUint16_t					 SEP_PrivExponentSize;
	DxUint16_t					 SEP_MaxPrivExponentSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Modulus_ptr;
	DxUint8_t					 SEP_KeyGenType;
	CRYS_RSA_HASH_OpMode_t       SEP_HashMode;
	DxUint16_t		             SEP_SaltLength;
	/* modulus size */
	DxUint16_t					 SEP_KeySize;
	DxUint16_t					 SEP_MaxKeySize = RSA_BufferMaxLength;
	DxUint8_t					 SEP_CRT_Mode;
	DxUint8_t					*SEP_P_ptr; 
	DxUint16_t					 SEP_PSize;
	DxUint16_t					 SEP_MaxPSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Q_ptr;
	DxUint16_t					 SEP_QSize;
	DxUint16_t					 SEP_MaxQSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_dP_ptr;
	DxUint16_t					 SEP_dPSize;
	DxUint16_t					 SEP_MaxdPSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_dQ_ptr;
	DxUint16_t					 SEP_dQSize;
	DxUint16_t					 SEP_MaxdQSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_qInv_ptr;
	DxUint16_t					 SEP_qInvSize;
	DxUint16_t					 SEP_MaxqInvSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Data_ptr;


	/* 1. extract parameters from the message */
	/******************************************/
	SEP_DataInSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_DATASIZE_IN_OFFSET];

	SEP_PubExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET];
	SEP_PrivExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET];
	/* modulus size */
	SEP_KeySize     = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_KEYSIZE_OFFSET];
	SEP_CRT_Mode    = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_CRTMODE_OFFSET];
	SEP_PSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PSIZE_OFFSET];
	SEP_QSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_QSIZE_OFFSET];
	SEP_dPSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dPSIZE_OFFSET];
	SEP_dQSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dQSIZE_OFFSET];
	SEP_qInvSize    = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_qInvSIZE_OFFSET];
	SEP_KeyGenType  = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_KEYGENTYPE_OFFSET];
	SEP_HashMode    = (CRYS_RSA_HASH_OpMode_t)msg[DX_EXT_SEP_HOST_RSA_HASHMODE_OFFSET];
	SEP_SaltLength  = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_SALTLENGTH_OFFSET];

	SEP_MaxDataInSize = ((SEP_DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);


	/* message length in words = length of the not buffer parameters */
	/* + all buffers parameters maximal lengthes                     */
	SEP_MsgLengthInWords = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS + 
		SEP_MaxDataInSize/4 + SEP_MaxPubExponentSize/4 + SEP_MaxPrivExponentSize/4 +
		SEP_MaxKeySize/4 + SEP_MaxPSize/4 + SEP_MaxQSize/4 + SEP_MaxdPSize/4 + 
		SEP_MaxdQSize/4 + SEP_MaxqInvSize/4 /*+ SEP_MaxExpectedDataSize/4*/;

	SEP_PubExponent_ptr  = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PubExponent_OFFSET];
	SEP_PrivExponent_ptr = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PrivExponent_OFFSET];
	SEP_Modulus_ptr		 = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Modulus_OFFSET];
	SEP_P_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_P_OFFSET];
	SEP_Q_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Q_OFFSET];
	SEP_dP_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dP_OFFSET];
	SEP_dQ_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dQ_OFFSET];
	SEP_qInv_ptr         =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_qInv_OFFSET];
	SEP_Data_ptr         = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_DataIn_OFFSET];


	/* 2. call QA SEP function */
	/***************************/
	SEP_Error = SEPQA_RSA_PSS_ExternalHash_SignVerify (	SEP_PubExponent_ptr,
														SEP_PubExponentSize,
														SEP_PrivExponent_ptr,
														SEP_Modulus_ptr,
														SEP_KeySize,
														SEP_CRT_Mode,
														SEP_KeyGenType,
														SEP_HashMode,
														SEP_SaltLength,
														SEP_P_ptr, 
														SEP_PSize,
														SEP_Q_ptr,
														SEP_QSize,
														SEP_dP_ptr, 
														SEP_dPSize,
														SEP_dQ_ptr,
														SEP_dQSize,
														SEP_qInv_ptr,
														SEP_qInvSize,
														SEP_Data_ptr,
														SEP_DataInSize,
														FailureFunctionNameOut);

	if(SEP_Error != CRYS_OK)
	{
		/* output parameters are placed after msg RC */
		DX_VOS_FastMemCpy((DxUint8_t  *)&msg[DX_SEP_HOST_SEP_PROTOCOL_MSG_RC_OFFSET + 1], (DxUint8_t  *)&FailureFunctionNameOut[0], TST_FUNC_STRING_LEN);
		SEP_MsgLengthInWords = TST_FUNC_STRING_LEN/sizeof(DxUint32_t);
	}

	/* 3. end SEP message and return */
	/**********************/
	return DX_SEP_EndMessage(msg, SEP_Error, SEP_MsgLengthInWords*sizeof(DxUint32_t));
}

/*****************************************************************************
* Function Name:                                                           
*  DX_SEP_PARSER_RSA_Get_PrivAndPubKey
*  
* Inputs:
*  msg - message with all requested parameters.
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
* This function does the parsing for the external application that executes acceptance test 
* for RSA PSS PKCS#1v2.1 integrated sign & Verify
* It calls the SEP function with parameters parsed from input message
* Algorithm: 
* 1. extract parameters from the message 
* 2. call QA SEP function
* 3. end SEP message and return
******************************************************************************/                                
DxUint32_t DX_SEP_PARSER_RSA_Get_PrivAndPubKey(DxUint32_t* msg)
{
	/********* Variables Definitions ************/
	DxUint32_t					SEP_Error = DX_SUCCESS;
	/* message length in words */
	DxUint32_t					SEP_MsgLengthInWords;

	DxUint8_t	                *SEP_PubExponent_ptr;
	DxUint16_t					 SEP_PubExponentSize = 0;
	DxUint16_t					 SEP_MaxPubExponentSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_PrivExponent_ptr;
	DxUint16_t					 SEP_PrivExponentSize;
	DxUint16_t					 SEP_MaxPrivExponentSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Modulus_ptr;
	DxUint16_t					 SEP_KeySize;
	DxUint16_t					 SEP_MaxKeySize = RSA_BufferMaxLength;
	DxUint8_t					 SEP_CRT_Mode;
	DxUint8_t					*SEP_P_ptr; 
	DxUint16_t					 SEP_PSize;
	DxUint16_t					 SEP_MaxPSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_Q_ptr;
	DxUint16_t					 SEP_QSize;
	DxUint16_t					 SEP_MaxQSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_dP_ptr;
	DxUint16_t					 SEP_dPSize;
	DxUint16_t					 SEP_MaxdPSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_dQ_ptr;
	DxUint16_t					 SEP_dQSize;
	DxUint16_t					 SEP_MaxdQSize = RSA_BufferMaxLength;
	DxUint8_t					*SEP_qInv_ptr;
	DxUint16_t					 SEP_qInvSize;
	DxUint16_t					 SEP_MaxqInvSize = RSA_BufferMaxLength;
	


	/* 1. extract parameters from the message */
	/******************************************/
	SEP_PubExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET];
	SEP_PrivExponentSize = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET];
	/* modulus size */
	SEP_KeySize     = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_KEYSIZE_OFFSET];
	SEP_CRT_Mode    = (DxUint8_t)msg[DX_EXT_SEP_HOST_RSA_CRTMODE_OFFSET];
	SEP_PSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_PSIZE_OFFSET];
	SEP_QSize       = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_QSIZE_OFFSET];
	SEP_dPSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dPSIZE_OFFSET];
	SEP_dQSize      = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_dQSIZE_OFFSET];
	SEP_qInvSize    = (DxUint16_t)msg[DX_EXT_SEP_HOST_RSA_qInvSIZE_OFFSET];

	/* message length in words = length of the not buffer parameters */
	/* + all buffers parameters maximal lengthes                     */
	SEP_MsgLengthInWords = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS + 
		/*SEP_MaxDataInSize/4 +*/ SEP_MaxPubExponentSize/4 + SEP_MaxPrivExponentSize/4 +
		SEP_MaxKeySize/4 + SEP_MaxPSize/4 + SEP_MaxQSize/4 + SEP_MaxdPSize/4 + 
		SEP_MaxdQSize/4 + SEP_MaxqInvSize/4 /*+ SEP_MaxExpectedDataSize/4*/;

	SEP_PubExponent_ptr  = (DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PubExponent_OFFSET];
	SEP_PrivExponent_ptr = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_PrivExponent_OFFSET];
	SEP_Modulus_ptr		 = 	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Modulus_OFFSET];
	SEP_P_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_P_OFFSET];
	SEP_Q_ptr            =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_Q_OFFSET];
	SEP_dP_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dP_OFFSET];
	SEP_dQ_ptr           =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_dQ_OFFSET];
	SEP_qInv_ptr         =	(DxUint8_t*)&msg[DX_EXT_SEP_HOST_RSA_qInv_OFFSET];


	/* 2. call QA SEP function */
	/***************************/
	SEP_Error = SEPQA_RSA_Get_PrivAndPubKey (SEP_PubExponent_ptr,
											 SEP_PubExponentSize,
											 SEP_PrivExponent_ptr,
											 SEP_PrivExponentSize,
											 SEP_Modulus_ptr,
											 SEP_KeySize,
											 SEP_CRT_Mode,
											 SEP_P_ptr, 
											 SEP_PSize,
											 SEP_Q_ptr,
											 SEP_QSize,
											 SEP_dP_ptr, 
											 SEP_dPSize,
											 SEP_dQ_ptr,
											 SEP_dQSize,
											 SEP_qInv_ptr,
											 SEP_qInvSize,
											 FailureFunctionNameOut);

	if(SEP_Error != CRYS_OK)
	{
		/* output parameters are placed after msg RC */
		DX_VOS_FastMemCpy((DxUint8_t  *)&msg[DX_SEP_HOST_SEP_PROTOCOL_MSG_RC_OFFSET + 1], (DxUint8_t  *)&FailureFunctionNameOut[0], TST_FUNC_STRING_LEN);
		SEP_MsgLengthInWords = TST_FUNC_STRING_LEN/sizeof(DxUint32_t);
	}

	/* 3. end SEP message and return */
	/**********************/
	return DX_SEP_EndMessage(msg, SEP_Error, SEP_MsgLengthInWords*sizeof(DxUint32_t));
}